unit mathimge;

interface

uses
  SysUtils, Winprocs, WinTypes,Messages, Classes, Graphics,
  Controls, Forms, Dialogs, ExtCtrls;

type
  colorarraytype=array[0..17] of longint;


  PD3FloatPoint=^TD3FloatPoint;
  TD3FloatPoint=record
              x,y,z:extended;
              next:PD3FloatPoint;
              end;
  {$IFDEF WINDOWS}
  PD3FloatPointArray=^TD3FloatPointarray;
  TD3FloatPointArray=array[0..300] of PD3FloatPoint;
  {$ENDIF}

  {Surface Object to be passed to D3DrawSurface:}
  TSurface=class(TObject)
    private
      Fxm,Fym:integer;
      {$IFDEF WINDOWS}
      FFloatsurface:array[0..300] of PD3FloatPointArray;
      {$ENDIF}
      {$IFDEF WIN32}
      FFloatsurface:array[0..300,0..300] of PD3FloatPoint;
      {$ENDIF}
    public
      Error:boolean;
      property xmesh:integer read Fxm;
      property ymesh:integer read Fym;
      constructor create(xgrid,ygrid:integer); virtual;
      {surface has (xgrid+1)*(ygrid+1) grid points.}
      {grids number from 0 to xgrid etc.}
      procedure Make(i,j:integer;x,y,z:extended);
      {assigns the point (x,y,z) to grid (i,j)}
      function D3Point(i,j:integer):TD3FloatPoint;
      {gets the point at grid (i,j)}
      destructor destroy; override;
    end;

  PFloatPoint=^TFloatpoint;
  TFloatPoint=record
     x,y:extended;
     next:PFloatPoint;
     end;

  {FloatPointList object to be passed to DrawPolyline:}
  {(Note: I find these lists easier to maintain than dynamic arrays)}
  TFloatPointList=class(TObject)
  private
    Fcount:longint;
  public
    Firstpoint,Currentpoint:PFloatpoint;
    next:TFloatPointList;
    property count:longint read Fcount;
    constructor create; virtual;
    destructor destroy; override;
    procedure add(x,y:extended);
    {add a point (x,y) at end of list}
    procedure assign(AFloatPointList:TFloatpointlist); virtual;
    {copy AFloatPointList to this one. AFloatPointList is still
    around as another instance.}
  end;

  {FloatPointListList object: list of FloatPointLists}
  TFloatPointListList=class(TObject)
  private
    FCount,FTotalCount:longint;
  public
    Firstlist,Currentlist:TFloatPointList;
    property count:longint read FCount;
    {number of *lists* in the object}
    property TotalCount:longint read FTotalcount;
    {number of total points in all lists}
    constructor create; virtual;
    destructor destroy; override;
    procedure add;
    {add a new list}
    procedure AddToCurrent(x,y:extended);
    {add (x,y) to the end of the current list.}
  end;

  TD3FloatPointList=class(TObject)
  private
    Fcount:longint;
  public
    Firstpoint,Currentpoint:PD3Floatpoint;
    next:TD3FloatPointList;
    property count:longint read Fcount;
    constructor create; virtual;
    destructor destroy; override;
    procedure add(x,y,z:extended);
    {add a point (x,y,z) at end of list}
    procedure assign(AFloatPointList:TD3Floatpointlist); virtual;
    {copy AFloatPointList to this one. AFloatPointList is still
    around as another instance.}
  end;

  {FloatPointListList object: list of FloatPointLists}
  TD3FloatPointListList=class(TObject)
  private
    FCount,FTotalCount:longint;
  public
    Firstlist,Currentlist:TD3FloatPointList;
    property count:longint read FCount;
    {number of *lists* in the object}
    property TotalCount:longint read FTotalcount;
    {number of total points in all lists}
    constructor create; virtual;
    destructor destroy; override;
    procedure add;
    {add a new list}
    procedure AddToCurrent(x,y,z:extended);
    {add (x,y) to the end of the current list.}
  end;



  TMathImage = class(timage)
  private
    x1d2,x2d2,y1d2,y2d2,ax,bx,ay,by:extended;
    axisglb:boolean;
    x1d3,x2d3,y1d3,y2d3,z1d3,z2d3,alpha:extended;
    zrd3,yrd3:extended;
    basex,basey,basez,frontx,fronty,frontz,vd:extended;
    arad,tana,thetaz,thetay,sinz,siny,cosz,cosy,
    ad3,bxd3,byd3,bzd3,ap,bxp,byp:extended;
    rightz, righty:extended;
    maxth,maxxtw,maxytw:integer;
    fversion:string;
    procedure setversion(x:string);
    procedure resetworld;
    procedure d3resetworld;
    procedure setaxisglb(a:boolean);
    procedure setx1d2(x:extended);
    procedure setx2d2(x:extended);
    procedure sety1d2(x:extended);
    procedure sety2d2(x:extended);
    procedure setx1d3(x:extended);
    procedure sety1d3(x:extended);
    procedure setx2d3(x:extended);
    procedure sety2d3(x:extended);
    procedure setz1d3(x:extended);
    procedure setz2d3(x:extended);
    procedure setvd(x:extended);
    procedure setzrd3(x:extended);
    procedure setyrd3(x:extended);
    procedure setalpha(x:extended);
    function scalar(xb,yb,zb:extended):extended;
    procedure blockx(x:extended;var xb:extended);
    procedure blocky(y:extended;var yb:extended);
    procedure blockz(z:extended;var zb:extended);
    procedure project(xb,yb,zb:extended; var u,v:extended);
    procedure makeradians;
    function dist(xb,yb,zb:extended):extended;
    procedure findbase(var i1,i2,i3:integer);
    procedure initworld;
    procedure drawoneaxis(x1,y1,z1,x2,y2,z2:extended;c:string);
    { Private declarations, never mind }
  protected
    { Protected declarations }
{---------------------*********************************--------------------------}
{                               THE IMPORTANT STUFF                                                 }
{---------------------*********************************--------------------------}
  public
    hregion:HRgn;
    {current clippingregion}

    property D2WorldX1: extended read x1d2 write setx1d2;
    property D2WorldX2: extended read x2d2 write setx2d2;
    property D2WorldY1: extended read y1d2 write sety1d2;
    property D2WorldY2: extended read y2d2 write sety2d2;
    {The above set the boundary for the 2-d-drawing world}

    property D2Axes:boolean read axisglb write setaxisglb;
    {If true, space is reserved on the image canvas to include axes.
    You need to call drawaxes in order to actually draw any.}

    property D3WorldX1: extended read x1d3 write setx1d3;
    property D3WorldX2: extended read x2d3 write setx2d3;
    property D3WorldY1: extended read y1d3 write sety1d3;
    property D3WorldY2: extended read y2d3 write sety2d3;
    property D3WorldZ1: extended read z1d3 write setz1d3;
    property D3WorldZ2: extended read z2d3 write setz2d3;
    {These set the boundaries for the 3-d-drawing world. When graphed,
     the world box is normalized so its longest side has length 2, and
     the other sides have lengthes according to the true aspect ratio of
     the bounds you specify. The box is then projected onto the image
     according to the settings of D3ViewDist, D3ViewAngle, D3Zrotation,
     D3yrotation}

    property D3Zrotation: extended read zrd3 write setzrd3;
    {Angle of viewpoint with the x-axis (how much it's rotated
     about the z-axis)}

    property D3Yrotation: extended read yrd3 write setyrd3;
    {Angle of viewpoint with the z-axis (how much the viewpoint is
     rotated about the y-axis)}

    property D3ViewDist: extended read vd write setvd;
    {Distance of viewpoint to the center of the d3-world, which has been
     normalized so its longest side has length 2. }

    property D3ViewAngle:extended read alpha write setalpha;
    {Opening angle of the lens of the viewpoint. Large D3ViewAngle combined with
     small D3ViewDist give fish eye effect. See Surface.pas to get a feel.}

    constructor create(AOwner:TComponent); override;
    destructor destroy; override;

    {The following are the methods for 2-d graphing:}
    procedure setworld(x1,y1,x2,y2:extended);
    {set world range in one step}

    procedure reset;
    {Call whenever the image is being resized. It adjusts the world to pixel
    scaling and resizes the bitmap}

    procedure setcolor(color:longint);
    {Short for canvas.pen.color:=color}

    function getcolor:longint;
    {Short for result:=canvas.pen.color;}

    function WindowX(X : extended):integer;
    {Translates World x to pixel x}

    function WindowY(Y : extended):integer;
    {Translates World y to pixel y}

    function worldx(xs:integer):extended;
    function worldy(ys:integer):extended;
    {Translate pixel to world}

    function norm(x,y:extended):extended;
    {Length of vector (x,y)}

    procedure clear;
    {Erases current picture, puts a new bitmap into picture}

    procedure DrawPoint(X, Y : extended);
   {puts a pixel with world coordinates (x,y) on the screen. Color
    is the currently selected pen color -> setcolor}

    procedure MovetoPoint(X,Y:extended);
   {moves the cursor to the point with world coordinates (x,y)}

    procedure DrawLine(X1, Y1, X2, Y2 : extended);
   {Draws a line from (x1,y1) to (x2,y2) in world coordinates}

    procedure DrawLineto(X,Y:extended);
   {draws a line from the current cursor position (see MovetoPoint) to
   point (x,y) in world coordinates}

    procedure DrawEllipse(x1,y1,x2,y2:extended);
    {draws an ellipse in the rectangle between (x1,y1) (lower left)
     and (x2,y2) (upper right) and fills it with the current brush color}

    procedure DrawRectangle(x1,y1,x2,y2:extended);
    {analog to canvas.rectangle}

    procedure DrawAxes(xlabel,ylabel:string;
                        zerolines:boolean;
                        axescolor,zerolinescolor:longint);
    {Draws Axes at the left and bottom boundary of the image. Ticks and
     labelling of numeric values are done automatically. xlabel, ylabel is
     text that goes to the end of the axes. Zerolines=true draws lines x=0,
     y=0. Axescolor,ZerolinesColor are selfexplaining.}

    procedure DrawVector(x,y,a,b:extended);
    {Draws a vector (a,b) at base (x,y)}

    procedure DrawPolyline(FloatPointList:TFloatpointlist);
    {Draws a line connecting points in FloatPointList. Faster
    than individual lines.}

    procedure DrawPolygon(FloatPointList:TFloatPointlist);
    {Draws a line connection the points in FloatPointList, closes the
    shape and fills it with the current brush)}

    procedure DrawPolyPolyLine(FloatPointListList:TFLoatPointListList);
    {Draws all lists in the ListList as Polylines}


   {D3Graphics procedures:}
    procedure d3setworld(x1,y1,z1,x2,y2,z2,vdist
          ,vangle:extended;zrot,yrot:extended);
    {Sets all d3-graphic parameters in one step}

    procedure d3window(x,y,z:extended; var xs,ys:integer);
    {translates world-(x,y,z) to pixel-(xs,ys)}

    procedure d3moveto(x,y,z:extended);
    procedure d3drawpoint(x,y,z:extended);
    procedure d3drawline(x1,y1,z1,x2,y2,z2:extended);
    procedure d3drawlineto(x,y,z:extended);
    {The 3-d-analogues of the line procedures}

    procedure d3drawaxes(c1,c2,c3:string);
    {Draws axes (without ticks yet) at the bondary of the world box
    and puts c1,c2,c3 on their ends.
    If somebody could come up with a good axis-labelling algorithm
    for 3D, I'd be very interested.}

    procedure d3drawworldbox;
    {Draws the box the current world resides in, with the 3 sides facing
    the viewer left open}

    procedure d3drawzerocross;
    {Draws lines x=y=0, x=z=0, y=z=0}

    procedure D3Polyline(FloatPointList:TD3Floatpointlist);

    procedure D3PolyPolyline(FloatPointListList:TD3Floatpointlistlist);
    {analogs to DrawPolyLine, DrawPolyPolyLine}

    function d3distancetoviewer(x,y,z:extended):extended;
    {The (scaled) distance of the viewpoint to point (x,y,z). Could be
    used to see whether something is visible, but it's a lengthy
    algorithm which I dumped.}

    {Surface Routines:}

    procedure d3DrawSurface(surface:TSurface; fill, NoUpdate:boolean);
    {Draw surface. Surface must habe been created and
    filled with the world coordinates of the gridpoints.
    Fill=false gives a wire frame, Fill=true displays it filled with
    the current brush color, invisible parts hidden. I've now found
    *the* cheap way to display filled surfaces right, previous was
    dumb.., but the way it's done now, it's not possible for a given
    surface point to *know* whether it's visible or not.
    NoUpdate=true: Display the surface all at once (fastest),
             false: Show some drawing steps in between. }

    { Public declarations }
  published
  property version:string read fversion write setversion;
  {Fake property to display the version of the component}
    { Published declarations }
  end;

const colorarray:colorarraytype = (clblack,clnavy,clmaroon,
 clolive,clpurple,clteal,clgray,clgreen,clred,clblue,clfuchsia,
 cllime,claqua,clyellow,clsilver,16777088,8454016,clwhite);
 {A handy array to circle through the system colors}

procedure Register;


implementation

procedure Register;
begin
  RegisterComponents('Samples', [TMathImage]);
end;

{TSurface}

constructor TSurface.create(xgrid,ygrid:integer);
var i,j:integer;
begin
  inherited create;
  Fxm:=-1; Fym:=-1;
  error:=true;
  if xgrid>=0 then if xgrid<=300 then if ygrid>=0 then if ygrid<=300
  then
  begin
    error:=false;
    Fxm:=xgrid; Fym:=ygrid;
    For i:=0 to Fxm do
    begin
      {$IFDEF WINDOWS}
      New(FFloatSurface[i]);
      {$ENDIF}
      For j:=0 to Fym do
      begin
        {$IFDEF WINDOWS}
        New(FFloatSurface[i]^[j]);
        with FFloatSurface[i]^[j]^ do
        {$ENDIF}
        {$IFDEF WIN32}
        New(FFloatSurface[i,j]);
        with FFloatSurface[i,j]^ do
        {$ENDIF}
        begin
          x:=0; y:=0; z:=0;
        end;
      end;
    end;
  end;
end;

procedure TSurface.Make(i,j:integer;x,y,z:extended);
begin
  if i>=0 then if i<=Fxm then if j>=0 then if j<=Fym
  then begin
    {$IFDEF WINDOWS}
    FFloatSurface[i]^[j]^.x:=x;
    FFloatSurface[i]^[j]^.y:=y;
    FFloatSurface[i]^[j]^.z:=z;
    {$ENDIF}
    {$IFDEF WIN32}
    FFloatSurface[i,j]^.x:=x;
    FFloatSurface[i,j]^.y:=y;
    FFloatSurface[i,j]^.z:=z;
    {$ENDIF}
  end;
end;

function TSurface.D3Point(i,j:integer):TD3FloatPoint;
var p:TD3FloatPoint;
begin
  if (i>=0) and (i<=Fxm) and (j>=0) and (j<=Fym) then
  {$IFDEF WINDOWS}
    p:=FFloatSurface[i]^[j]^
  {$ENDIF}
  {$IFDEF WIN32}
    p:=FFloatSurface[i,j]^
  {$ENDIF}
  else with p do
  begin
    x:=0; y:=0; z:=0;
  end;
  result:=p;
end;

destructor TSurface.destroy;
var i,j:integer;
begin
  for i:=0 to Fxm do
  begin
    for j:=0 to Fym do
    {$IFDEF WINDOWS}
    dispose(FFloatSurface[i]^[j]);
    dispose(FFloatSurface[i]);
    {$ENDIF}
    {$IFDEF WIN32}
    dispose(FFLoatSurface[i,j]);
    {$ENDIF};
  end;
  inherited destroy;
end;

{TFloatPointList}

constructor TFloatPointList.create;
begin
  inherited create;
  Firstpoint:=nil;
  Fcount:=0;
  Currentpoint:=nil;
  next:=nil;
end;

procedure TFloatPointList.add(x,y:extended);
var p:PFloatPoint;
begin
  new(p);
  p^.x:=x; p^.y:=y; p^.next:=nil;
  inc(Fcount);
  if Firstpoint=nil then
  begin
    Firstpoint:=p;
    Currentpoint:=p;
  end
  else
  begin
    currentpoint^.next:=p;
    currentpoint:=p;
  end;
end;

procedure TFloatPointList.assign;
var p:PFloatpoint; i:longint;
begin
  If AFloatPointlist.count>0 then
  begin
    p:=AFloatPointList.firstpoint;
    for i:=1 to AFloatPointlist.count do
    begin
      add(p^.x,p^.y);
      p:=p^.next;
    end;
  end;
end;


destructor TFloatPointList.destroy;
var p,q:PFloatPoint;
begin
  if FirstPoint<>nil then
  begin
    p:=Firstpoint;
    while p^.next<>nil do
    begin
      q:=p^.next;
      dispose(p);
      p:=q;
    end;
    dispose(p);
  end;
  firstpoint:=nil;
  Currentpoint:=nil;
  inherited destroy;
end;


{TFloatPointListList}

constructor TFloatPointListList.create;
begin
  inherited create;
  Firstlist:=nil;
  Fcount:=0;
  FTotalCount:=0;
  Currentlist:=nil;
end;

procedure TFloatPointListList.add;
var p:TFloatPointList;
begin
  p:=TFloatPointList.create;
  inc(Fcount);
  if Firstlist=nil then
  begin
    Firstlist:=p;
    Currentlist:=p;
  end
  else
  begin
    currentlist.next:=p;
    currentlist:=p;
  end;
end;

procedure TFloatPointListList.AddToCurrent(x,y:extended);
var p:PFloatPoint;
begin
  currentlist.add(x,y);
  inc(FTotalcount);
end;

destructor TFloatPointListList.destroy;
var p,q:TFloatPointList;
begin
  if FirstList<>nil then
  begin
    p:=FirstList;
    while p.next<>nil do
    begin
      q:=p.next;
      p.free;
      p:=q;
    end;
    p.free;
  end;
  firstlist:=nil;
  Currentlist:=nil;
  inherited destroy;
end;

{TD3FloatPointList}

constructor TD3FloatPointList.create;
begin
  inherited create;
  Firstpoint:=nil;
  Fcount:=0;
  Currentpoint:=nil;
  next:=nil;
end;

procedure TD3FloatPointList.add(x,y,z:extended);
var p:PD3FloatPoint;
begin
  new(p);
  p^.x:=x; p^.y:=y; p^.z:=z; p^.next:=nil;
  inc(Fcount);
  if Firstpoint=nil then
  begin
    Firstpoint:=p;
    Currentpoint:=p;
  end
  else
  begin
    currentpoint^.next:=p;
    currentpoint:=p;
  end;
end;

procedure TD3FloatPointList.assign;
var p:PD3Floatpoint; i:longint;
begin
  If AFloatPointlist.count>0 then
  begin
    p:=AFloatPointList.firstpoint;
    for i:=1 to AFloatPointlist.count do
    begin
      add(p^.x,p^.y,p^.z);
      p:=p^.next;
    end;
  end;
end;


destructor TD3FloatPointList.destroy;
var p,q:PD3FloatPoint;
begin
  if FirstPoint<>nil then
  begin
    p:=Firstpoint;
    while p^.next<>nil do
    begin
      q:=p^.next;
      dispose(p);
      p:=q;
    end;
    dispose(p);
  end;
  firstpoint:=nil;
  Currentpoint:=nil;
  inherited destroy;
end;


{TD3FloatPointListList}

constructor TD3FloatPointListList.create;
begin
  inherited create;
  Firstlist:=nil;
  Fcount:=0;
  FTotalCount:=0;
  Currentlist:=nil;
end;

procedure TD3FloatPointListList.add;
var p:TD3FloatPointList;
begin
  p:=TD3FloatPointList.create;
  inc(Fcount);
  if Firstlist=nil then
  begin
    Firstlist:=p;
    Currentlist:=p;
  end
  else
  begin
    currentlist.next:=p;
    currentlist:=p;
  end;
end;

procedure TD3FloatPointListList.AddToCurrent(x,y,z:extended);
var p:PD3FloatPoint;
begin
  currentlist.add(x,y,z);
  inc(FTotalcount);
end;

destructor TD3FloatPointListList.destroy;
var p,q:TD3FloatPointList;
begin
  if FirstList<>nil then
  begin
    p:=FirstList;
    while p.next<>nil do
    begin
      q:=p.next;
      p.free;
      p:=q;
    end;
    p.free;
  end;
  firstlist:=nil;
  Currentlist:=nil;
  inherited destroy;
end;



{TMathImage}
procedure TMathImage.setversion;
begin
  fversion:=x;
end;

procedure TMathImage.setx1d2;
begin
  x1d2:=x; resetworld;
end;
procedure TMathImage.setx2d2;
begin
  x2d2:=x; resetworld;
end;
procedure TMathImage.sety1d2;
begin
  y1d2:=x; resetworld;
end;
procedure TMathImage.sety2d2;
begin
  y2d2:=x; resetworld;
end;
procedure TMathImage.setx1d3;
begin
  x1d3:=x; d3resetworld;
end;
procedure TMathImage.setx2d3;
begin
  x2d3:=x; d3resetworld;
end;
procedure TMathImage.sety1d3;
begin
  y1d3:=x; d3resetworld;
end;
procedure TMathImage.sety2d3;
begin
  y2d3:=x; d3resetworld;
end;
procedure TMathImage.setz1d3;
begin
  z1d3:=x; d3resetworld;
end;
procedure TMathImage.setz2d3;
begin
  z2d3:=x; d3resetworld;
end;
procedure TMathImage.setvd;
begin
  vd:=x; d3resetworld;
end;
procedure TMathImage.setalpha;
begin
  alpha:=x; d3resetworld;
end;
procedure TMathImage.setzrd3;
begin
  zrd3:=x; d3resetworld;
end;
procedure TMathImage.setyrd3;
begin
  yrd3:=x; d3resetworld;
end;

Constructor TMathImage.Create(AOwner:TComponent);
var num:integer;  bitmap:tbitmap;
begin
  inherited create(AOWner);
  controlstyle:=controlstyle+[csopaque];
  hregion:=0;
  x1d2:=-1; x2d2:=1; y1d2:=-1; y2d2:=1;
  x1d3:=-1; x2d3:=1; y1d3:=-1; y2d3:=1;
  z1d3:=-1; z2d3:=1; axisglb:=false; alpha:=6;
  vd:=6.4; zrd3:=45; yrd3:=45;
  fversion:='2.1 July 97 ';
end;



Destructor TMathImage.Destroy;
var i,j:integer;
begin
  if hregion<>0 then deleteobject(hregion);
  inherited destroy;
end;

procedure TMathImage.setaxisglb;
var r:Trect;
begin
  axisglb:=a;
  resetworld;
  if a then
  begin
    hregion:=CreateRectRgn(windowx(x1d2)+1,windowy(y2d2),
         windowx(x2d2),windowy(y1d2)-1);
    SelectClipRgn(canvas.handle,hregion);
  end else
  begin
    deleteobject(hregion);
    hregion:=0;
    SelectClipRgn(canvas.handle,0);
  end;
end;

procedure TMathImage.setworld;
 procedure xerror;
 begin
   if not (csDesigning in ComponentState) then
     application.messagebox('Error: d2worldx1>=d2worldx2','MathImage Error',mb_OK);
     x2d2:=x1d2+1;
 end;

 procedure yerror;
 begin
    if not (csDesigning in ComponentState) then
     application.messagebox('Error: d2worldy1>=d2worldy2','MathImage Error',MB_OK);
     y2d2:=y1d2+1;
 end;
 function max(i,j:integer):integer;
 begin
   if i<j then result:=j else result:=i;
 end;

 function maxtextwidth(xx1,xx2:extended):integer;
 var itemp,xtick:extended;
     i,istart,ticks,w:integer;
 begin
   canvas.font.size:=canvas.font.size-1;
   {There's one pointer in a 1.0-VCL here that doesn't get freed.
    But what's one little pointer..
   Didn't have any tool to test 2.0-leakage...}
   itemp:=ln((xx2-xx1)/8)/ln(10);
   if itemp>=0 then i:=trunc(itemp) else i:=trunc(itemp)-1;
   xtick:=exp(i*ln(10));
   istart:=round(xx1/xtick);
   while istart*xtick<xx1 do inc(istart);
   ticks:=round((xx2-x1)/xtick);
   w:=canvas.textwidth(floattostrf(istart*xtick,ffgeneral,3,3));
   for i:=1 to 4 do
   w:=max(w,canvas.textwidth(floattostrf((istart+i*(ticks div 4))*xtick,ffgeneral,3,3)));
   result:=w;
   canvas.font.size:=canvas.font.size+1;
 end;
begin
  x1d2:=x1; x2d2:=x2; y1d2:=y1; y2d2:=y2;
  If x2d2<=x1d2 then xerror;
  If y2d2<=y1d2 then yerror;
  maxytw:=maxtextwidth(y1d2,y2d2);
  maxxtw:=maxtextwidth(x1d2,x2d2);
  maxth:=canvas.textheight('-1.234567');
  if axisglb then
    Bx := (width-7-maxytw-(maxxtw div 2))/(X2d2 - X1d2)
  else
    Bx:=(width-1)/(x2d2-x1d2);
  if axisglb then
  Ax :=6+maxytw-X1d2*Bx else
  Ax :=  - X1d2 * Bx;
  If y2d2<=y1d2 then yerror;
  if axisglb then
  By:=(height-7-maxth-(maxth div 2))/(y1d2-Y2d2) else
  By :=(height-1) / (Y1d2 - Y2d2);
  if axisglb then
  Ay:=(maxth div 2)-By*Y2d2 else
  Ay := - Y2d2 * By;
end;

procedure TMathImage.resetworld;
begin
  setworld(x1d2,y1d2,x2d2,y2d2);
end;

procedure TMathImage.reset;
begin
  resetworld;
  d3resetworld;
  picture.bitmap.height:=height;
  picture.bitmap.width:=width;
end;

procedure TMathImage.setcolor;
begin
  canvas.pen.color:=color;
end;

function TMathImage.getcolor;
begin
  result:=canvas.pen.color;
end;

function TMathImage.WindowX;
var Temp: extended;
begin
  Temp := Ax + Bx * X;
  if abs(temp)<6000 then
    result:= round(Temp)
  else
    if temp<0 then result:=-6000 else result:=6000;
end;

function TMathImage.WindowY;
var Temp : extended;
begin
  Temp := Ay + By * Y;
  if abs(temp)<6000 then
    result:= round(Temp)
  else
    if temp<0 then result:=-6000 else result:=6000;
end;

function TMathImage.norm;
begin
  result:=sqrt(sqr(x)+sqr(y));
end;

function TMathImage.worldx;
begin
  result:=(xs-Ax)/Bx;
end;

function TMathImage.worldy;
begin
  result:=(ys-ay)/by;
end;

procedure TMathImage.clear;
var bitmap:tbitmap;
begin
  bitmap:=tbitmap.create;
  bitmap.width:=width; bitmap.height:=height;
  picture.graphic:=bitmap;
  bitmap.free; {This line should not be here according to
                the Delphi documentation, but without it
                the component leaks memory.}
  {canvas.fillrect(rect(0,0,width,height));}
  {The commented line above also clears the picture
   but leads to another memory leak}
  invalidate;
end;

procedure TMathImage.DrawPoint;
begin
   SelectClipRgn(canvas.handle,hregion);
  canvas.pixels[windowx(x),windowy(y)]:=canvas.pen.color;
end;

procedure TMathImage.MovetoPoint;
begin
  SelectClipRgn(canvas.handle,hregion);
  canvas.moveto(windowx(x),windowy(y));
end;

procedure TMathImage.DrawLine;
var xw,yw:integer;
begin
  SelectClipRgn(canvas.handle,hregion);
  xw:=windowx(x1); yw:=windowy(y1);
  canvas.polyline([point(xw,yw),
                   point(windowx(x2),windowy(y2)),
                   point(xw,yw)]);
end;


procedure TMathImage.DrawLineto(x,y:extended);
begin
  SelectClipRgn(canvas.handle,hregion);
  canvas.Lineto(windowx(x),windowy(y));
end;

procedure TMathImage.DrawEllipse(x1,y1,x2,y2:extended);
begin
  SelectClipRgn(canvas.handle,hregion);
  canvas.ellipse(windowx(x1),windowy(y2),windowx(x2),windowy(y1));
end;

procedure TMathImage.DrawRectangle;
begin
  SelectClipRgn(canvas.handle,hregion);
  canvas.rectangle(windowx(x1),windowy(y2),windowx(x2)+1,windowy(y1)+1);
  {Note: the one is added to make the rectangle come out the same as
  it would by using the canvas.polygon method. I don't know why the
  standard canvas rectangle is always a pixel too short.}
end;

procedure TMathImage.drawaxes;
var xs,ys:integer; i,istart,ticks,savecolor:longint;
    t:string; itemp,xtick,ytick:extended;
begin
  savecolor:=canvas.pen.color;
  setaxisglb(true);
  deleteobject(hregion);
  hregion:=0;
  selectclipRgn(canvas.handle,0);
  canvas.pen.color:=axescolor;
  drawline(x1d2,y1d2,x2d2,y1d2);
  itemp:=ln((d2worldx2-d2worldx1)/8)/ln(10);
  if itemp>=0 then
   i:=trunc(itemp) else i:=trunc(itemp)-1;
  xtick:=exp(i*ln(10));
  itemp:=ln((d2worldy2-d2worldy1)/8)/ln(10);
  if itemp>=0 then
    i:=trunc(itemp) else i:=trunc(itemp)-1;
  ytick:=exp(i*ln(10));
  if xtick>0 then
  begin
    istart:=round(x1d2/xtick);
    while istart*xtick<x1d2 do inc(istart);
    i:=istart;
    ticks:=round((x2d2-x1d2)/xtick);
    with canvas.font do
     size:=size-1;
    if ticks<=2000 then
    repeat
      xs:=windowx(i*xtick);
      ys:=windowy(y1d2);
      canvas.moveto(xs,ys);
      canvas.lineto(xs,ys+4);
      if (i-istart) mod (ticks div 4) =0 then
      begin
        t:=floattostrf(i*xtick,ffgeneral,3,3);
        with canvas do
        begin
          textout(xs-(textwidth(t) div 2), ys+6,t);
          moveto(xs,ys);
          lineto(xs,ys+6);
        end;
      end;
      inc(i)
    until i*xtick>x2d2;
  end;
  with canvas.font do size:=size+1;
  xs:=windowx(x2d2);
  ys:=windowy(y1d2);
  canvas.moveto(xs,ys);
  canvas.lineto(xs-6,ys-6);
  canvas.moveto(xs,ys);
  canvas.lineto(xs-6,ys+6);
  canvas.textout(xs-canvas.textwidth(xlabel)-4,ys-canvas.textheight(xlabel)-6,xlabel);
  drawline(x1d2,y1d2,x1d2,y2d2);
  if ytick>0 then
  begin
    istart:=round(y1d2/ytick);
    while istart*ytick<y1d2 do inc(istart);
    i:=istart;
    ticks:=round((y2d2-y1d2)/ytick);
    with canvas.font do
      size:=size-1;
    if ticks <=2000 then
    repeat
      xs:=windowx(x1d2);
      ys:=windowy(i*ytick);
      canvas.moveto(xs,ys);
      canvas.lineto(xs-4,ys);
      if (i-istart) mod (ticks div 4) =0 then
      begin
        t:=floattostrf(i*ytick,ffgeneral,3,3);
        with canvas do
        begin
          textout(xs-textwidth(t)-6,ys-textheight(t) div 2,t);
          moveto(xs,ys);
          lineto(xs-6,ys);
        end;
      end;
      inc(i);
    until i*ytick>y2d2;
  end;
  with canvas.font do
    size:=size+1;
  xs:=windowx(x1d2);
  ys:=windowy(y2d2);
  canvas.moveto(xs,ys);
  canvas.lineto(xs+6,ys+6);
  canvas.moveto(xs,ys);
  canvas.lineto(xs-6,ys+6);
  canvas.textout(xs+8,ys,ylabel);
  if zerolines then
  begin
    canvas.pen.color:=zerolinescolor;
    drawline(0,y1d2,0,y2d2);
    drawline(x1d2,0,x2d2,0);
  end;
  canvas.pen.color:=savecolor;
  setaxisglb(true);
end;

procedure TMathImage.drawvector;
var aw,bw,xw,yw,u1,u2,v1,v2:integer; n:extended;
begin
  SelectClipRgn(canvas.handle,hregion);
  v1:=windowx(a+x);
  v2:=windowy(b+y);
  xw:=windowx(x);
  yw:=windowy(y);
  aw:=v1-xw;
  bw:=v2-yw;
  n:=norm(bw-aw,aw+bw);
  if n>0 then
  begin
  canvas.moveto(xw,yw);
  canvas.lineto(v1,v2);
  u1:=round(8.0*(bw-aw)/n);
  u2:=round(8.0*(-bw-aw)/n);
  canvas.moveto(v1,v2);
  canvas.lineto(v1+u1,v2+u2);
  u1:=round(8.0*(-aw-bw)/n);
  u2:=round(8.0*(aw-bw)/n);
  canvas.moveto(v1,v2);
  canvas.lineto(v1+u1,v2+u2);
  end;
end;


procedure TMathimage.DrawPolyline(FloatPointList:TFloatpointlist);
{$IFDEF WINDOWS}
type  TPointArray=array[0..16320] of TPoint;
{$ENDIF}
{$IFDEF WIN32}
type TPointArray=array[0..1000000] of TPoint;
{$ENDIF}
var p:PFloatPoint; i:longint; pointarray:^TPointArray;
begin
  SelectClipRgn(canvas.handle,hregion);
  with FloatPointList do
  begin
    {$IFDEF WINDOWS}
    if count>16320 then begin {Output error message} exit; end;
    {$ENDIF}
    getmem(pointarray,count*SizeOf(TPoint));
    p:=firstpoint;
    for i:=1 to count do
    with pointarray^[i-1] do
    begin
      x:=windowx(p^.x); y:=windowy(p^.y);
      p:=p^.next;
    end;
    picture.bitmap.canvas.pen.mode:=canvas.pen.mode;
    polyline(canvas.handle,pointarray^,count);
    invalidate;
    freemem(pointarray,count*SizeOf(TPoint));
  end;
end;

procedure TMathimage.DrawPolygon(FloatPointList:TFloatpointlist);
{$IFDEF WINDOWS}
type  TPointArray=array[0..16320] of TPoint;
{$ENDIF}
{$IFDEF WIN32}
type TPointArray=array[0..1000000] of TPoint;
{$ENDIF}
var p:PFloatPoint; i:longint; pointarray:^TPointArray;
begin
  SelectClipRgn(canvas.handle,hregion);
  with FloatPointList do
  begin
    {$IFDEF WINDOWS}
    if count>16320 then begin {Output error message} exit; end;
    {$ENDIF}
    getmem(pointarray,count*SizeOf(TPoint));
    p:=firstpoint;
    for i:=1 to count do
    with pointarray^[i-1] do
    begin
      x:=windowx(p^.x); y:=windowy(p^.y);
      p:=p^.next;
    end;
    picture.bitmap.canvas.pen.mode:=canvas.pen.mode;
    polygon(canvas.handle,pointarray^,count);
    invalidate;
    freemem(pointarray,count*SizeOf(TPoint));
  end;
end;

procedure TMathimage.DrawPolyPolyline(FloatPointListList:TFloatpointlistlist);
{$IFDEF WIN32}
type  TPointArray=array[0..1000000] of TPoint;
      TCountArray=array[0..1000000] of Integer;
var p:PFloatPoint; q:TFloatPointList; i,j,k:longint;
        pointarray:^TPointArray; countarray:^TCountArray;
begin
  SelectClipRgn(canvas.handle,hregion);
  with FloatPointListList do
  begin
    getmem(pointarray,totalcount*SizeOf(TPoint));
    getmem(countarray,count*SizeOf(Integer));
    q:=firstlist; k:=0;
    for i:=1 to count do
    begin
      p:=q.firstpoint;
      for j:=1 to  q.count do
      begin
        with pointarray^[k] do
        begin
          x:=windowx(p^.x); y:=windowy(p^.y);
        end;
        inc(k);
        p:=p^.next;
      end;
      countarray^[i-1]:=q.count;
      q:=q.next;
    end;
    polypolyline(canvas.handle,pointarray^,countarray^,count);
    invalidate;
    freemem(pointarray,totalcount*SizeOf(TPoint));
    freemem(countarray,count*SizeOf(Integer));
  end;
end;
{$ENDIF}

{$IFDEF WINDOWS}
var q:TFloatPointList; i:longint;
begin
  If FloatPointListList.count>0 then
  with FloatPointListList do
  begin
    q:=Firstlist;
    For i:=1 to count do
    begin
      DrawPolyline(q);
      q:=q.next;
    end;
  end;
end;
{$ENDIF}



function max(x,y:extended):extended;
begin
  if x<y then result:=y else result:=x;
end;

function min(x,y:extended):extended;
begin
  if x<y then result:=x else result:=y;
end;


procedure TMathImage.makeradians;
  procedure d3worlderror;
  begin
    if x1d3>=x2d3 then
    begin
      application.messagebox('Error: d3worldx1>=d3worldx2','MathImage Error',mb_OK);
      x2d3:=x1d3+1;
    end;
    if y1d3>=y2d3 then
    begin
      application.messagebox('Error: d3worldy1>=d3worldy2','MathImage Error',mb_OK);
      y2d3:=y1d3+1;
    end;
    if z1d3>=z2d3 then
    begin
      application.messagebox('Error: d3worldz1>=d3worldz2','MathImage Error',mb_OK);
      z2d3:=z1d3+1;
    end;
    ad3:=max(max(x2d3-x1d3,y2d3-y1d3),z2d3-z1d3);
  end;

begin
  thetaz:=2*pi*zrd3/360;
  thetay:=2*pi*yrd3/360;
  arad:=pi*alpha/360;
  sinz:=sin(thetaz); cosz:=cos(thetaz);
  siny:=sin(thetay); cosy:= cos(thetay);
  tana:=sin(arad)/cos(arad);
  rightz:=(zrd3+90) - 180*trunc((zrd3+90.0)/180);
  righty:=yrd3 - 180*trunc(yrd3/180);
  ad3:=max(max(x2d3-x1d3,y2d3-y1d3),z2d3-z1d3);
  if ad3<=0 then d3worlderror;
  ad3:=2/ad3;
  bxd3:=-ad3*(x1d3+x2d3)/2;
  byd3:=-ad3*(y1d3+y2d3)/2;
  bzd3:=-ad3*(z1d3+z2d3)/2;
  ap:=min(height,width)/2/tana/vd;
  bxp:=width/2; byp:=height/2;
end;

function TMathImage.scalar(xb,yb,zb:extended):extended;
begin
  scalar:=yb*sinz*siny+zb*cosy+xb*siny*cosz;
end;

function TMathImage.dist(xb,yb,zb:extended):extended;
begin
  dist:=d3viewdist-scalar(xb,yb,zb);
end;

function TMathImage.d3distancetoviewer(x,y,z:extended):extended;
var xb,yb,zb:extended;
begin
  blockx(x,xb); blocky(y,yb); blockz(z,zb);
  d3distancetoviewer:=sqrt(sqr(d3viewdist*siny*sinz-yb)+
    sqr(d3viewdist*cosy-zb)+sqr(d3viewdist*siny*cosz-xb));
end;


procedure TMathImage.findbase(var i1,i2,i3:integer);
var dmax,d:extended; i,j,k:integer;
begin
  i1:=-1;i2:=-1;i3:=-1;
  dmax:=0;
  for i:=0 to 1 do
  for j:=0 to 1 do
  for k:=0 to 1 do
  begin
    d:=dist(-1+2*i,-1+2*j,-1+2*k);
    dmax:=max(dmax,d);
    if d=dmax then
    begin
      i1:=-1+2*i;i2:=-1+2*j;i3:=-1+2*k;
    end;
  end;
end;

procedure TMathImage.initworld;
var umin,umax,vmin,vmax,d2w:extended;
    i1,i2,i3:integer;
    i,j,k:integer;
begin
  if d3viewdist<0 then d3viewdist:=0.0000001;
  if alpha > 179 then alpha:=179;
  if alpha <0.1 then alpha:=0.1;
  makeradians;
  findbase(i1,i2,i3);
    if i1=-1 then basex:=x1d3 else basex:=x2d3;
    if i2=-1 then basey:=y1d3 else basey:=y2d3;
    if i3=-1 then basez:=z1d3 else basez:=z2d3;
    if i1=1 then frontx:=x1d3 else frontx:=x2d3;
    if i2=1 then fronty:=y1d3 else fronty:=y2d3;
    if i3=1 then frontz:=z1d3 else frontz:=z2d3;
end;

procedure TMathImage.d3setworld;
var d:extended;
begin
  x1d3:=x1;
  x2d3:=x2;
  y2d3:=y2;
  y1d3:=y1;
  y2d3:=y2;
  z1d3:=z1;
  z2d3:=z2;
  zrd3:=zrot; yrd3:=yrot; vd:=vdist; alpha:=vangle;
  initworld;
end;

procedure TMathImage.d3resetworld;
begin
  d3setworld(x1d3,y1d3,z1d3,x2d3,y2d3,z2d3,
     vd,alpha,zrd3,yrd3);
end;

procedure TMathImage.blockx(x:extended;var xb:extended);
begin
  xb:=bxd3+ad3*x;
end;

procedure TMathImage.blocky(y:extended;var yb:extended);
begin
  yb:=byd3+ad3*y;
end;

procedure TMathImage.blockz(z:extended;var zb:extended);
begin
  zb:=bzd3+ad3*z;
end;

procedure TMathImage.d3window(x,y,z:extended; var xs,ys:integer);
var xb,yb,zb,rad,tan,tempx,tempy,d,u,v:extended;
begin
  blockx(x,xb);
  blocky(y,yb);
  blockz(z,zb);
  project(xb,yb,zb,u,v);
  tempx:=bxp+ap*u;
  if abs(tempx)<6000 then xs:=round(tempx)
  else if tempx<0 then xs:=-6000 else xs:=6000;
  tempy:=byp-ap*v;
  if abs(tempy)<6000 then ys:=round(tempy)
  else if tempy <0 then ys:=-6000 else ys:=6000;
end;

procedure TMathImage.project;
var scal,d:extended;
begin
  scal:=scalar(xb,yb,zb);
  d:=d3viewdist-scal;
  if righty<>0 then
      v:=(zb-scal*cosy)/siny
  else
      v:=-(yb*sinz+xb*cosz)/cosy;
  if rightz<>0 then
  u:=(Yb+sinz*(v*cosy-scal*siny))/cosz
    else
      u:=-Xb*sinz;
  if d<=0 then d:=1.e-10;
  u:=u/d;
  v:=v/d;
end;

procedure TMathImage.d3moveto(x,y,z:extended);
var xs,ys:integer; visible:boolean;
begin
  d3window(x,y,z,xs,ys);
  canvas.moveto(xs,ys);
end;

procedure TMathImage.d3drawpoint(x,y,z:extended);
var xs,ys:integer;
begin
  d3window(x,y,z,xs,ys);
  canvas.pixels[xs,ys]:=canvas.pen.color;
end;

procedure TMathImage.d3drawline(x1,y1,z1,x2,y2,z2:extended);
var u1,v1,u2,v2:integer;
begin
  d3window(x1,y1,z1,u1,v1);
  d3window(x2,y2,z2,u2,v2);
  canvas.moveto(u1,v1);
  canvas.lineto(u2,v2);
  canvas.pixels[u2,v2]:=canvas.pen.color;
end;

procedure TMathImage.d3drawlineto(x,y,z:extended);
var xs,ys:integer;
begin
    d3window(x,y,z,xs,ys);
    canvas.lineto(xs,ys);
end;


procedure TMathImage.drawoneaxis(x1,y1,z1,x2,y2,z2:extended;c:string);
var norms,wx,wy:extended;
    xs1,ys1,xs2,ys2:integer; vsx,vsy:extended;
begin
  d3drawline(x1,y1,z1,x2,y2,z2);
  d3window(x1,y1,z1,xs1,ys1);
  d3window(x2,y2,z2,xs2,ys2);
  vsx:=(xs2-xs1); vsy:=(ys2-ys1);
  norms:=sqrt(vsx*vsx+vsy*vsy);
  if norms>0 then
  begin
    vsx:=vsx/norms; vsy:=vsy/norms;
    wx:=(-vsx+vsy)/sqrt(2); wy:=(-vsy-vsx)/sqrt(2);
    canvas.moveto(xs2,ys2);
    canvas.lineto(xs2+round(5*wx),ys2+round(5*wy));
    wx:=(-vsx-vsy)/sqrt(2); wy:=(-vsy+vsx)/sqrt(2);
    canvas.moveto(xs2,ys2);
    canvas.lineto(xs2+round(5*wx),ys2+round(5*wy));
    canvas.textout(xs2-10,ys2-10,c);
  end;
end;


procedure TMathImage.d3drawaxes(c1,c2,c3:string);


begin   {******* drawd3axes ******}
    drawoneaxis(x1d3,y1d3,z1d3,x2d3,y1d3,z1d3,c1);
    drawoneaxis(x1d3,y1d3,z1d3,x1d3,y2d3,z1d3,c2);
    drawoneaxis(x1d3,y1d3,z1d3,x1d3,y1d3,z2d3,c3);
end;

procedure TMathImage.d3drawzerocross;
begin
   if 0>=x1d3 then if 0<=x2d3 then if 0>=z1d3 then if 0<=z2d3 then
   d3drawline(0,y1d3,0,0,y2d3,0);
   if 0>=z1d3 then if 0<=z2d3 then if 0>=y1d3 then if 0<=y2d3 then
   d3drawline(x1d3,0,0,x2d3,0,0);
   if 0>=y1d3 then if 0<=y2d3 then if 0>=x1d3 then if 0<=x2d3 then
   d3drawline(0,0,z1d3,0,0,z2d3);
end;

procedure TMathImage.d3drawworldbox;
var i:integer; delta:extended; savestyle:tpenstyle;
savecolor:tcolor;
begin
  savestyle:=canvas.pen.style;
  canvas.pen.style:=pssolid;
  savecolor:=canvas.pen.color;
  canvas.pen.color:=clblack;
    d3drawline(basex,basey,basez,frontx,basey,basez);
    d3drawline(basex,basey,basez,basex,fronty,basez);
    d3drawline(basex,basey,basez,basex,basey,frontz);
    d3drawline(basex,fronty,basez,frontx,fronty,basez);
    d3drawline(basex,fronty,basez,basex,fronty,frontz);
    d3drawline(basex,basey,frontz,frontx,basey,frontz);
    d3drawline(basex,basey,frontz,basex,fronty,frontz);
    d3drawline(frontx,basey,basez,frontx,fronty,basez);
    d3drawline(frontx,basey,basez,frontx,basey,frontz);
 canvas.pen.style:=savestyle;
 canvas.pen.color:=savecolor;
end;

procedure TMathimage.D3Polyline(FloatPointList:TD3Floatpointlist);
{$IFDEF WINDOWS}
type  TPointArray=array[0..16320] of TPoint;
{$ENDIF}
{$IFDEF WIN32}
type TPointArray=array[0..1000000] of TPoint;
{$ENDIF}
var p:PD3FloatPoint; xw,yw:integer; i:longint; pointarray:^TPointArray;
begin
  SelectClipRgn(canvas.handle,hregion);
  with FloatPointList do
  begin
    {$IFDEF WINDOWS}
    if count>16320 then begin {Output error message} exit; end;
    {$ENDIF}
    getmem(pointarray,count*SizeOf(TPoint));
    p:=firstpoint;
    for i:=1 to count do
    with pointarray^[i-1] do
    begin
      D3window(p^.x,p^.y,p^.z,xw,yw);
      x:=xw; y:=yw;
      p:=p^.next;
    end;
    polyline(canvas.handle,pointarray^,count);
    invalidate;
    freemem(pointarray,count*SizeOf(TPoint));
  end;
end;

procedure TMathimage.D3PolyPolyline(FloatPointListList:TD3Floatpointlistlist);
{$IFDEF WIN32}
type  TPointArray=array[0..1000000] of TPoint;
      TCountArray=array[0..1000000] of Integer;
var p:PD3FloatPoint; q:TD3FloatPointList; xw,yw:integer;
    i,j,k:longint;
    pointarray:^TPointArray;
    countarray:^TCountArray;
begin
  SelectClipRgn(canvas.handle,hregion);
  with FloatPointListList do
  begin
    getmem(pointarray,totalcount*SizeOf(TPoint));
    getmem(countarray,count*SizeOf(Integer));
    q:=firstlist; k:=0;
    for i:=1 to count do
    begin
      p:=q.firstpoint;
      for j:=1 to  q.count do
      begin
        with pointarray^[k] do
        begin
          d3window(p^.x,p^.y,p^.z,xw,yw);
          x:=xw; y:=yw;
        end;
        inc(k);
        p:=p^.next;
      end;
      countarray^[i-1]:=q.count;
      q:=q.next;
    end;
    polypolyline(canvas.handle,pointarray^,countarray^,count);
    invalidate;
    freemem(pointarray,totalcount*SizeOf(TPoint));
    freemem(countarray,count*SizeOf(Integer));
  end;
end;
{$ENDIF}

{$IFDEF WINDOWS}
var q:TD3FloatPointList; i:longint;
begin
  If FloatPointListList.count>0 then
  with FloatPointListList do
  begin
    q:=Firstlist;
    For i:=1 to count do
    begin
      D3Polyline(q);
      q:=q.next;
    end;
  end;
end;
{$ENDIF}


{Surface}

type
     PD3Point=^TD3Point;
     TD3point=record
             pt:TPoint;
             dist:single;
            end;
     {$IFDEF WINDOWS}
     PD3PointArray=^TD3PointArray;
     TD3PointArray=array[0..300] of PD3Point;
     {$ENDIF}

     PCell=^TCell;
     TCell=record
             vertex:array[0..3] of PD3Point;
             dist:single;
           end;
     TListCompare=function(item1,item2:pointer):integer;

function Compare(item1,item2:pointer):integer; far;
var Cell1,Cell2:PCell;
begin
  Cell1:=PCell(item1);
  Cell2:=PCell(item2);
  if Cell1^.dist>Cell2^.dist then result:=-1
  else if Cell1^.dist<Cell2^.dist then result:=1
  else result:=0;
end;

{$IFDEF WINDOWS}
procedure sort(var AList:Tlist; compare: TListCompare);

  procedure QuickSort(var AList:TList; compare: TListCompare; iLo,iHi:integer);
  var Lo, Hi:integer; Mid: pointer;
  begin
    with AList do
    begin
      Lo := iLo;
      Hi := iHi;
      Mid := items[(Lo + Hi) div 2];
      repeat
        while Compare(items[Lo],Mid)<0 do Inc(Lo);
        while Compare(items[Hi],Mid)>0 do Dec(Hi);
        if Lo <= Hi then
        begin
          Exchange(Lo,Hi);
          Inc(Lo);
          Dec(Hi);
        end;
      until Lo > Hi;
      if Hi > iLo then QuickSort(AList,Compare,iLo, Hi);
      if Lo < iHi then QuickSort(AList,Compare, Lo, iHi);
    end;
  end;
begin
  Quicksort(Alist,Compare,0,Alist.count-1);
end;
{$ENDIF}

procedure TMathImage.d3drawsurface(surface:TSurface; fill,NoUpdate:boolean);
var i,j,xw,yw:integer;
    color,savecolor:longint;
    Alist:Tlist;
    ACell:PCell;
    Apoint:TD3FloatPoint;
    p0,p1,p2,p3:PD3Point;
    {$IFDEF WINDOWS}
    Screensurface:array[0..300] of PD3PointArray;
    {$ENDIF}
    {$IFDEF WIN32}
    ScreenSurface:array[0..300,0..300] of PD3Point;
    {$ENDIF}
begin
  if not surface.error then
  with surface do
  begin
    for i:=0 to xmesh do
    begin
      {$IFDEF WINDOWS}
      new(screensurface[i]);
      {$ENDIF}
      for j:=0 to ymesh do
      begin
        Apoint:=D3Point(i,j);
        {$IFDEF WINDOWS}
        new(screensurface[i]^[j]);
        with screensurface[i]^[j]^ do
        {$ENDIF}
        {$IFDEF WIN32}
        new(screensurface[i,j]);
        with screensurface[i,j]^ do
        {$ENDIF}
        with Apoint do
        begin
          D3Window(x,y,z,xw,yw);
          pt.x:=xw; pt.y:=yw;
          dist:=d3distancetoviewer(x,y,z);
        end;
      end;
    end;
    if fill then
    begin
      Alist:=Tlist.create;
      for i:=0 to xmesh-1 do
      for j:=0 to ymesh-1 do
      begin
        if not NoUpdate then application.processmessages;
        New(ACell);
        p0:={$IFDEF WINDOWS}screensurface[i]^[j];{$ENDIF}
            {$IFDEF WIN32}  screensurface[i,j];{$ENDIF}
        p1:={$IFDEF WINDOWS}screensurface[i+1]^[j];{$ENDIF}
            {$IFDEF WIN32}  screensurface[i+1,j];{$ENDIF}
        p2:={$IFDEF WINDOWS}screensurface[i+1]^[j+1];{$ENDIF}
            {$IFDEF WIN32}  screensurface[i+1,j+1];{$ENDIF}
        p3:={$IFDEF WINDOWS}screensurface[i]^[j+1];{$ENDIF}
             {$IFDEF WIN32} screensurface[i,j+1];{$ENDIF}
        ACell^.vertex[0]:=p0;
        ACell^.vertex[1]:=p1;
        ACell^.vertex[2]:=p2;
        ACell^.vertex[3]:=p3;
        ACell^.dist:=(p0^.dist+p1^.dist+p2^.dist+p3^.dist)/4;
        AList.add(ACell);
      end;
      {$IFDEF WINDOWS}
      Sort(Alist,Compare);
      {$ENDIF}
      {$IFDEF WIN32}
      Alist.Sort(Compare);
      {$ENDIF}
      with AList do
      begin
        repaint;
        for i:=0 to count-1 do
        begin
          ACell:=PCell(items[i]);
          with ACell^ do
          canvas.polygon([vertex[0]^.pt,vertex[1]^.pt,vertex[2]^.pt,vertex[3]^.pt]);
          if not NoUpdate then if i mod 20 =0 then
          begin
            repaint;
            application.processmessages;
          end;
        end;
        for i:=0 to count-1 do
        begin
          Acell:=PCell(items[i]);
          dispose(Acell);
        end;
      end;
      Alist.free;
    end {if fill}
    else
    begin
      for i:=0 to xmesh do
      begin
        if not NoUpdate then
        begin
          repaint;
          application.processmessages;
        end;
        for j:=0 to ymesh do
        begin
          {$IFDEF WINDOWS}
          with screensurface[i]^[j]^ do
          canvas.moveto(pt.x,pt.y);
          if j<ymesh then
          with screensurface[i]^[j+1]^ do
          canvas.lineto(pt.x,pt.y);
          with screensurface[i]^[j]^ do
          canvas.moveto(pt.x,pt.y);
          if i<xmesh then
          with screensurface[i+1]^[j]^ do
          canvas.lineto(pt.x,pt.y);
          {$ENDIF}
          {$IFDEF WIN32}
          with screensurface[i,j]^ do
          canvas.moveto(pt.x,pt.y);
          if j<ymesh then
          with screensurface[i,j+1]^ do
          canvas.lineto(pt.x,pt.y);
          with screensurface[i,j]^ do
          canvas.moveto(pt.x,pt.y);
          if i<xmesh then
          with screensurface[i+1,j]^ do
          canvas.lineto(pt.x,pt.y);
          {$ENDIF}
        end; {for j}
      end; {for i}
    end; {if not fill}
    for i:=0 to xmesh do
    begin
      for j:=0 to ymesh do
      {$IFDEF WINDOWS}
      Dispose(ScreenSurface[i]^[j]);
      Dispose(ScreenSurface[i]);
      {$ENDIF}
      {$IFDEF WIN32}
      Dispose(ScreenSurface[i,j]);
      {$ENDIF}
    end;
  end; {if not surface.error}
end;

end.
